import { z } from "zod"
import DOMPurify from "dompurify"

const sanitize = (val: string) => DOMPurify.sanitize(val.trim())

export const hotelPartnerSchema = z.object({
  hotelName: z
    .string()
    .min(2, "Hotel name must be at least 2 characters")
    .max(100, "Hotel name is too long")
    .transform(sanitize),
  ownerName: z
    .string()
    .min(2, "Owner name must be at least 2 characters")
    .max(100, "Owner name is too long")
    .transform(sanitize),
  stateId: z.string().min(1, "Please select a state"),
  cityId: z.string().min(1, "Please select a city"),
  email: z.string().email("Please enter a valid email").transform(sanitize),
  phone: z
    .string()
    .min(10, "Phone number must be at least 10 digits")
    .max(15, "Phone number is too long")
    .regex(/^[+]?[\d\s-]+$/, "Please enter a valid phone number")
    .transform(sanitize),
  password: z
    .string()
    .min(8, "Password must be at least 8 characters")
    .max(50, "Password is too long")
    .regex(/[A-Z]/, "Password must contain at least one uppercase letter")
    .regex(/[a-z]/, "Password must contain at least one lowercase letter")
    .regex(/[0-9]/, "Password must contain at least one number")
    .optional()
    .or(z.literal("")),
  address: z
    .string()
    .min(10, "Address must be at least 10 characters")
    .max(300, "Address is too long")
    .transform(sanitize),
  gstNumber: z
    .string()
    .min(15, "GST number must be 15 characters")
    .max(15, "GST number must be 15 characters")
    .regex(/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/, "Please enter a valid GST number")
    .transform(sanitize),
  panNumber: z
    .string()
    .min(10, "PAN number must be 10 characters")
    .max(10, "PAN number must be 10 characters")
    .regex(/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/, "Please enter a valid PAN number")
    .transform(sanitize),
  registrationNumber: z
    .string()
    .min(5, "Registration number is required")
    .max(50, "Registration number is too long")
    .transform(sanitize),
  licenseNumber: z
    .string()
    .min(5, "License number is required")
    .max(50, "License number is too long")
    .transform(sanitize),
  otherDocuments: z
    .string()
    .optional()
    .transform((val) => (val ? sanitize(val) : "")),
  since: z.string().min(1, "Please select partner since date"),
})

export const createHotelPartnerSchema = hotelPartnerSchema.extend({
  password: z
    .string()
    .min(8, "Password must be at least 8 characters")
    .max(50, "Password is too long")
    .regex(/[A-Z]/, "Password must contain at least one uppercase letter")
    .regex(/[a-z]/, "Password must contain at least one lowercase letter")
    .regex(/[0-9]/, "Password must contain at least one number"),
})

export type HotelPartnerInput = z.infer<typeof hotelPartnerSchema>
export type CreateHotelPartnerInput = z.infer<typeof createHotelPartnerSchema>
