// Admin Settings Data Store
export interface SMTPSettings {
  host: string
  port: number
  username: string
  password: string
  encryption: "none" | "ssl" | "tls"
  fromEmail: string
  fromName: string
}

export interface CashfreeSettings {
  appId: string
  secretKey: string
  environment: "sandbox" | "production"
  webhookSecret: string
}

export interface AgentCommissionSettings {
  defaultCommissionRate: number
  minimumBookingValue: number
  commissionType: "percentage" | "fixed"
  payoutCycle: "weekly" | "biweekly" | "monthly"
}

export interface WithdrawSettings {
  minimumWithdrawAmount: number
  maximumWithdrawAmount: number
  processingFee: number
  processingFeeType: "percentage" | "fixed"
  autoApproveThreshold: number
  withdrawalLockPeriod: number
}

export interface WithdrawMethod {
  id: string
  name: string
  type: "bank_transfer" | "upi" | "paypal" | "crypto"
  isActive: boolean
  requiredFields: string[]
}

export interface AdminSettings {
  smtp: SMTPSettings
  cashfree: CashfreeSettings
  agentCommission: AgentCommissionSettings
  withdraw: WithdrawSettings
  withdrawMethods: WithdrawMethod[]
}

// Default settings
const defaultSettings: AdminSettings = {
  smtp: {
    host: "smtp.gmail.com",
    port: 587,
    username: "",
    password: "",
    encryption: "tls",
    fromEmail: "noreply@teertham.com",
    fromName: "Teertham",
  },
  cashfree: {
    appId: "",
    secretKey: "",
    environment: "sandbox",
    webhookSecret: "",
  },
  agentCommission: {
    defaultCommissionRate: 10,
    minimumBookingValue: 1000,
    commissionType: "percentage",
    payoutCycle: "monthly",
  },
  withdraw: {
    minimumWithdrawAmount: 500,
    maximumWithdrawAmount: 100000,
    processingFee: 2,
    processingFeeType: "percentage",
    autoApproveThreshold: 5000,
    withdrawalLockPeriod: 7,
  },
  withdrawMethods: [
    {
      id: "bank_transfer",
      name: "Bank Transfer",
      type: "bank_transfer",
      isActive: true,
      requiredFields: ["accountNumber", "ifscCode", "accountHolderName", "bankName"],
    },
    {
      id: "upi",
      name: "UPI",
      type: "upi",
      isActive: true,
      requiredFields: ["upiId"],
    },
    {
      id: "paypal",
      name: "PayPal",
      type: "paypal",
      isActive: false,
      requiredFields: ["paypalEmail"],
    },
  ],
}

class SettingsDataStore {
  private settings: AdminSettings = { ...defaultSettings }

  getSettings(): AdminSettings {
    return { ...this.settings }
  }

  updateSMTP(data: Partial<SMTPSettings>): SMTPSettings {
    this.settings.smtp = { ...this.settings.smtp, ...data }
    return this.settings.smtp
  }

  updateCashfree(data: Partial<CashfreeSettings>): CashfreeSettings {
    this.settings.cashfree = { ...this.settings.cashfree, ...data }
    return this.settings.cashfree
  }

  updateAgentCommission(data: Partial<AgentCommissionSettings>): AgentCommissionSettings {
    this.settings.agentCommission = { ...this.settings.agentCommission, ...data }
    return this.settings.agentCommission
  }

  updateWithdraw(data: Partial<WithdrawSettings>): WithdrawSettings {
    this.settings.withdraw = { ...this.settings.withdraw, ...data }
    return this.settings.withdraw
  }

  updateWithdrawMethod(id: string, data: Partial<WithdrawMethod>): WithdrawMethod | undefined {
    const index = this.settings.withdrawMethods.findIndex((m) => m.id === id)
    if (index === -1) return undefined
    this.settings.withdrawMethods[index] = { ...this.settings.withdrawMethods[index], ...data }
    return this.settings.withdrawMethods[index]
  }

  addWithdrawMethod(method: Omit<WithdrawMethod, "id">): WithdrawMethod {
    const newMethod: WithdrawMethod = {
      ...method,
      id: `method_${Date.now()}`,
    }
    this.settings.withdrawMethods.push(newMethod)
    return newMethod
  }

  deleteWithdrawMethod(id: string): boolean {
    const index = this.settings.withdrawMethods.findIndex((m) => m.id === id)
    if (index === -1) return false
    this.settings.withdrawMethods.splice(index, 1)
    return true
  }
}

export const settingsDataStore = new SettingsDataStore()
