"use client"

import { useState } from "react"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Bell, Calendar, IndianRupee, AlertCircle, Info, Trash2, CheckCheck, Clock } from "lucide-react"

interface Notification {
  id: string
  type: "booking" | "payment" | "system" | "alert"
  title: string
  message: string
  time: string
  isRead: boolean
}

const notificationsData: Notification[] = [
  {
    id: "1",
    type: "booking",
    title: "New Booking Received",
    message: "Rahul Sharma has booked Deluxe Room for Dec 10-12, 2024. Total: ₹9,000",
    time: "5 minutes ago",
    isRead: false,
  },
  {
    id: "2",
    type: "payment",
    title: "Withdrawal Approved",
    message: "Your withdrawal request of ₹75,000 has been approved and transferred to HDFC Bank XXXX1234",
    time: "2 hours ago",
    isRead: false,
  },
  {
    id: "3",
    type: "alert",
    title: "Guest Check-out Reminder",
    message: "3 guests are scheduled for check-out today. Please prepare their bills.",
    time: "5 hours ago",
    isRead: false,
  },
  {
    id: "4",
    type: "booking",
    title: "Booking Cancelled",
    message: "Vikram Singh cancelled their booking for Dec 15-18. Refund of ₹5,000 initiated.",
    time: "1 day ago",
    isRead: true,
  },
  {
    id: "5",
    type: "system",
    title: "Profile Update Required",
    message: "Please update your hotel's GSTIN number for tax compliance.",
    time: "2 days ago",
    isRead: true,
  },
  {
    id: "6",
    type: "payment",
    title: "Weekly Earnings Summary",
    message: "Your earnings this week: ₹1,25,000 from 15 bookings. View detailed report.",
    time: "3 days ago",
    isRead: true,
  },
  {
    id: "7",
    type: "booking",
    title: "Stay Extended",
    message: "Amit Kumar extended their stay in Suite 501 by 2 more days. Additional: ₹24,000",
    time: "4 days ago",
    isRead: true,
  },
  {
    id: "8",
    type: "alert",
    title: "Low Room Availability",
    message: "Only 2 Deluxe Rooms available for next weekend. Consider adjusting prices.",
    time: "5 days ago",
    isRead: true,
  },
]

const typeConfig = {
  booking: { icon: Calendar, color: "bg-blue-100 text-blue-600 dark:bg-blue-950 dark:text-blue-400" },
  payment: { icon: IndianRupee, color: "bg-emerald-100 text-emerald-600 dark:bg-emerald-950 dark:text-emerald-400" },
  system: { icon: Info, color: "bg-violet-100 text-violet-600 dark:bg-violet-950 dark:text-violet-400" },
  alert: { icon: AlertCircle, color: "bg-amber-100 text-amber-600 dark:bg-amber-950 dark:text-amber-400" },
}

export function NotificationsContent() {
  const [notifications, setNotifications] = useState(notificationsData)
  const [filter, setFilter] = useState<"all" | "unread">("all")

  const unreadCount = notifications.filter((n) => !n.isRead).length

  const filteredNotifications = filter === "all" ? notifications : notifications.filter((n) => !n.isRead)

  const markAsRead = (id: string) => {
    setNotifications((prev) => prev.map((n) => (n.id === id ? { ...n, isRead: true } : n)))
  }

  const markAllAsRead = () => {
    setNotifications((prev) => prev.map((n) => ({ ...n, isRead: true })))
  }

  const deleteNotification = (id: string) => {
    setNotifications((prev) => prev.filter((n) => n.id !== id))
  }

  const clearAllRead = () => {
    setNotifications((prev) => prev.filter((n) => !n.isRead))
  }

  return (
    <div className="max-w-3xl mx-auto space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold">Notifications</h1>
          <p className="text-muted-foreground">
            {unreadCount > 0
              ? `You have ${unreadCount} unread notification${unreadCount > 1 ? "s" : ""}`
              : "All caught up!"}
          </p>
        </div>
        <div className="flex items-center gap-2">
          {unreadCount > 0 && (
            <Button variant="outline" size="sm" onClick={markAllAsRead}>
              <CheckCheck className="mr-2 h-4 w-4" />
              Mark all read
            </Button>
          )}
          <Button
            variant="outline"
            size="sm"
            onClick={clearAllRead}
            className="text-red-600 hover:text-red-700 bg-transparent hover:bg-red-50"
          >
            <Trash2 className="mr-2 h-4 w-4" />
            Clear read
          </Button>
        </div>
      </div>

      {/* Filter Tabs */}
      <div className="flex gap-2">
        <Button
          variant={filter === "all" ? "default" : "outline"}
          size="sm"
          onClick={() => setFilter("all")}
          className={filter === "all" ? "bg-slate-900 hover:bg-slate-800 dark:bg-white dark:text-slate-900" : ""}
        >
          All ({notifications.length})
        </Button>
        <Button
          variant={filter === "unread" ? "default" : "outline"}
          size="sm"
          onClick={() => setFilter("unread")}
          className={filter === "unread" ? "bg-slate-900 hover:bg-slate-800 dark:bg-white dark:text-slate-900" : ""}
        >
          Unread ({unreadCount})
        </Button>
      </div>

      {/* Notifications List */}
      <div className="space-y-3">
        {filteredNotifications.length === 0 ? (
          <Card className="border-0 shadow-sm">
            <CardContent className="py-12 text-center">
              <Bell className="mx-auto h-12 w-12 text-muted-foreground/50" />
              <h3 className="mt-4 font-semibold">No notifications</h3>
              <p className="text-sm text-muted-foreground">
                {filter === "unread" ? "All notifications have been read" : "You're all caught up!"}
              </p>
            </CardContent>
          </Card>
        ) : (
          filteredNotifications.map((notification) => {
            const config = typeConfig[notification.type]
            const Icon = config.icon
            return (
              <Card
                key={notification.id}
                className={`border shadow-sm hover:shadow-md transition-all cursor-pointer ${
                  !notification.isRead
                    ? "bg-slate-50 dark:bg-slate-900 border-l-4 border-l-slate-700 dark:border-l-white"
                    : ""
                }`}
                onClick={() => markAsRead(notification.id)}
              >
                <CardContent className="p-4">
                  <div className="flex gap-4">
                    <div className={`p-2 rounded-full h-fit ${config.color}`}>
                      <Icon className="h-5 w-5" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-start justify-between gap-2">
                        <div>
                          <h3
                            className={`font-medium ${!notification.isRead ? "text-foreground" : "text-muted-foreground"}`}
                          >
                            {notification.title}
                          </h3>
                          <p className="text-sm text-muted-foreground mt-1">{notification.message}</p>
                        </div>
                        <Button
                          variant="ghost"
                          size="icon"
                          className="h-8 w-8 text-muted-foreground hover:text-red-600 hover:bg-red-50"
                          onClick={(e) => {
                            e.stopPropagation()
                            deleteNotification(notification.id)
                          }}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                      <div className="flex items-center gap-2 mt-2">
                        <Clock className="h-3 w-3 text-muted-foreground" />
                        <span className="text-xs text-muted-foreground">{notification.time}</span>
                        {!notification.isRead && (
                          <Badge className="bg-slate-200 text-slate-700 dark:bg-slate-700 dark:text-slate-200 text-xs">
                            New
                          </Badge>
                        )}
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )
          })
        )}
      </div>
    </div>
  )
}
