"use client"

import { useState, useMemo } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Card, CardContent } from "@/components/ui/card"
import { CalendarCheck, PlayCircle, CheckCircle2, XCircle, IndianRupee, Car } from "lucide-react"
import { vehicleBookingDataStore } from "@/lib/data/vehicle-bookings"
import { VehicleBookingTable } from "./vehicle-booking-table"

type TabType = "booked" | "ongoing" | "completed" | "cancelled"

export function VehicleBookingsClient() {
  const [activeTab, setActiveTab] = useState<TabType>("booked")

  const stats = useMemo(() => vehicleBookingDataStore.getBookingStats(), [])

  const bookings = useMemo(() => {
    switch (activeTab) {
      case "booked":
        return vehicleBookingDataStore.getBookedBookings()
      case "ongoing":
        return vehicleBookingDataStore.getOngoingBookings()
      case "completed":
        return vehicleBookingDataStore.getCompletedBookings()
      case "cancelled":
        return vehicleBookingDataStore.getCancelledBookings()
      default:
        return []
    }
  }, [activeTab])

  // Format currency
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  return (
    <section aria-labelledby="vehicle-bookings-heading" className="admin-section">
      {/* Header */}
      <header className="admin-page-header">
        <h1 id="vehicle-bookings-heading" className="admin-page-title">
          Vehicle Bookings
        </h1>
        <p className="admin-page-description">View and manage all vehicle bookings on the platform.</p>
      </header>

      <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-5 mb-6">
        <Card className="admin-card rounded-xl border-0">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Total Bookings</p>
                <p className="text-2xl font-bold">{stats.total}</p>
                <p className="text-xs text-muted-foreground mt-1">All time bookings</p>
              </div>
              <div className="h-12 w-12 rounded-xl bg-muted/50 flex items-center justify-center">
                <Car className="h-6 w-6 text-muted-foreground" aria-hidden="true" />
              </div>
            </div>
          </CardContent>
        </Card>
        <Card className="admin-card rounded-xl border-0">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Booked</p>
                <p className="text-2xl font-bold text-blue-600">{stats.booked}</p>
                <p className="text-xs text-muted-foreground mt-1">{formatCurrency(stats.bookedAmount)} pending</p>
              </div>
              <div className="h-12 w-12 rounded-xl bg-blue-500/10 flex items-center justify-center">
                <CalendarCheck className="h-6 w-6 text-blue-600" aria-hidden="true" />
              </div>
            </div>
          </CardContent>
        </Card>
        <Card className="admin-card rounded-xl border-0">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Ongoing</p>
                <p className="text-2xl font-bold text-amber-600">{stats.ongoing}</p>
                <p className="text-xs text-muted-foreground mt-1">{formatCurrency(stats.ongoingAmount)} in progress</p>
              </div>
              <div className="h-12 w-12 rounded-xl bg-amber-500/10 flex items-center justify-center">
                <PlayCircle className="h-6 w-6 text-amber-600" aria-hidden="true" />
              </div>
            </div>
          </CardContent>
        </Card>
        <Card className="admin-card rounded-xl border-0">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Completed</p>
                <p className="text-2xl font-bold text-green-600">{stats.completed}</p>
                <p className="text-xs text-muted-foreground mt-1">{formatCurrency(stats.completedAmount)} earned</p>
              </div>
              <div className="h-12 w-12 rounded-xl bg-green-500/10 flex items-center justify-center">
                <CheckCircle2 className="h-6 w-6 text-green-600" aria-hidden="true" />
              </div>
            </div>
          </CardContent>
        </Card>
        <Card className="admin-card rounded-xl border-0">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Total Revenue</p>
                <p className="text-2xl font-bold text-primary">{formatCurrency(stats.totalAmount)}</p>
                <p className="text-xs text-muted-foreground mt-1">{stats.cancelled} cancelled</p>
              </div>
              <div className="h-12 w-12 rounded-xl bg-primary/10 flex items-center justify-center">
                <IndianRupee className="h-6 w-6 text-primary" aria-hidden="true" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={(v) => setActiveTab(v as TabType)} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="booked"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            <CalendarCheck className="mr-2 h-4 w-4" />
            Booked ({stats.booked})
          </TabsTrigger>
          <TabsTrigger
            value="ongoing"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            <PlayCircle className="mr-2 h-4 w-4" />
            Ongoing ({stats.ongoing})
          </TabsTrigger>
          <TabsTrigger
            value="completed"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            <CheckCircle2 className="mr-2 h-4 w-4" />
            Completed ({stats.completed})
          </TabsTrigger>
          <TabsTrigger
            value="cancelled"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            <XCircle className="mr-2 h-4 w-4" />
            Cancelled ({stats.cancelled})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="booked" className="mt-0">
          <VehicleBookingTable bookings={bookings} />
        </TabsContent>
        <TabsContent value="ongoing" className="mt-0">
          <VehicleBookingTable bookings={bookings} />
        </TabsContent>
        <TabsContent value="completed" className="mt-0">
          <VehicleBookingTable bookings={bookings} />
        </TabsContent>
        <TabsContent value="cancelled" className="mt-0">
          <VehicleBookingTable bookings={bookings} />
        </TabsContent>
      </Tabs>
    </section>
  )
}
