"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Separator } from "@/components/ui/separator"
import { MapPin, Car, Hotel } from "lucide-react"
import type { SpiritualPackage } from "@/lib/data/spiritual-packages"

interface ViewPackageDetailsDialogProps {
  open: boolean
  onOpenChange: (open: boolean) => void
  packageData: SpiritualPackage
}

export function ViewPackageDetailsDialog({ open, onOpenChange, packageData }: ViewPackageDetailsDialogProps) {
  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <div className="flex items-start justify-between">
            <div>
              <DialogTitle className="text-2xl">{packageData.cityName}</DialogTitle>
              <p className="text-sm text-muted-foreground mt-1">
                {packageData.totalLocations} locations • {packageData.dayPlans.length} days
              </p>
            </div>
            <Badge variant={packageData.status === "active" ? "default" : "secondary"} className="capitalize">
              {packageData.status}
            </Badge>
          </div>
        </DialogHeader>

        <div className="space-y-6 mt-4">
          {packageData.dayPlans.map((day) => (
            <Card key={day.dayNumber}>
              <CardHeader>
                <CardTitle className="text-lg">Day {day.dayNumber}</CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* Locations */}
                <div>
                  <div className="flex items-center gap-2 mb-3">
                    <MapPin className="h-4 w-4 text-primary" />
                    <h4 className="font-semibold">Locations ({day.locations.length})</h4>
                  </div>
                  <div className="grid gap-4 sm:grid-cols-2">
                    {day.locations.map((location) => (
                      <Card key={location.id} className="overflow-hidden">
                        <img
                          src={location.image || "/placeholder.svg"}
                          alt={location.name}
                          className="h-32 w-full object-cover"
                        />
                        <CardContent className="p-3">
                          <h5 className="font-medium mb-1">{location.name}</h5>
                          <p className="text-xs text-muted-foreground line-clamp-2">{location.itineraryDescription}</p>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                </div>

                <Separator />

                {/* Vehicles */}
                <div>
                  <div className="flex items-center gap-2 mb-3">
                    <Car className="h-4 w-4 text-primary" />
                    <h4 className="font-semibold">Vehicles ({day.vehicles.length})</h4>
                  </div>
                  <div className="grid gap-3 sm:grid-cols-2">
                    {day.vehicles.map((vehicle) => (
                      <Card key={vehicle.id} className="p-3">
                        <div className="flex items-center gap-3">
                          <img
                            src={vehicle.vehiclePhoto || "/placeholder.svg"}
                            alt={vehicle.vehicleName}
                            className="h-16 w-20 rounded object-cover"
                          />
                          <div>
                            <p className="font-medium text-sm">{vehicle.vehicleName}</p>
                            <p className="text-xs text-muted-foreground">₹{vehicle.perDayPrice}/day</p>
                          </div>
                        </div>
                      </Card>
                    ))}
                  </div>
                </div>

                {/* Hotels */}
                {day.hotels.length > 0 && (
                  <>
                    <Separator />
                    <div>
                      <div className="flex items-center gap-2 mb-3">
                        <Hotel className="h-4 w-4 text-primary" />
                        <h4 className="font-semibold">Hotels ({day.hotels.length})</h4>
                      </div>
                      <div className="grid gap-3 sm:grid-cols-2">
                        {day.hotels.map((hotel) => (
                          <Card key={hotel.id} className="p-3">
                            <div className="flex items-center gap-3">
                              <img
                                src={hotel.hotelPhoto || "/placeholder.svg"}
                                alt={hotel.hotelName}
                                className="h-16 w-20 rounded object-cover"
                              />
                              <div>
                                <p className="font-medium text-sm">{hotel.hotelName}</p>
                              </div>
                            </div>
                          </Card>
                        ))}
                      </div>
                    </div>
                  </>
                )}

                <Separator />

                {/* Pricing */}
                <div>
                  <h4 className="font-semibold mb-3">Pricing</h4>
                  <div className="grid gap-3 sm:grid-cols-3">
                    <Card className="p-3">
                      <p className="text-xs text-muted-foreground">B2C Weekday</p>
                      <p className="text-lg font-bold">₹{day.pricing.b2cWeekdayPerHead}</p>
                      <p className="text-xs text-muted-foreground">per head</p>
                    </Card>
                    <Card className="p-3">
                      <p className="text-xs text-muted-foreground">B2C Weekend</p>
                      <p className="text-lg font-bold">₹{day.pricing.b2cWeekendPerHead}</p>
                      <p className="text-xs text-muted-foreground">per head</p>
                    </Card>
                    <Card className="p-3">
                      <p className="text-xs text-muted-foreground">B2B Agent</p>
                      <p className="text-lg font-bold">₹{day.pricing.b2bAgentPricing}</p>
                      <p className="text-xs text-muted-foreground">pricing</p>
                    </Card>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </DialogContent>
    </Dialog>
  )
}
