"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import type { FixPackage } from "@/lib/data/fix-packages"
import { Calendar, MapPin, IndianRupee, Info } from "lucide-react"

interface ViewPackageDetailsDialogProps {
  package: FixPackage | null
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ViewPackageDetailsDialog({ package: pkg, open, onOpenChange }: ViewPackageDetailsDialogProps) {
  if (!pkg) return null

  // Format currency
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  // Calculate discount percentage
  const discountPercentage =
    pkg.originalPrice > pkg.discountedPrice
      ? Math.round(((pkg.originalPrice - pkg.discountedPrice) / pkg.originalPrice) * 100)
      : 0

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <div className="flex items-start justify-between gap-4">
            <div>
              <DialogTitle className="text-2xl">{pkg.name}</DialogTitle>
              <div className="flex items-center gap-2 mt-2">
                <Badge variant={pkg.status === "active" ? "default" : "secondary"}>
                  {pkg.status === "active" ? "Active" : "Inactive"}
                </Badge>
                {discountPercentage > 0 && (
                  <Badge variant="outline" className="bg-green-50 text-green-700 border-green-200">
                    {discountPercentage}% OFF
                  </Badge>
                )}
              </div>
            </div>
          </div>
        </DialogHeader>

        <div className="space-y-6 mt-4">
          {/* Pricing Section */}
          <div className="flex items-center gap-4 p-4 bg-muted/50 rounded-lg">
            <IndianRupee className="h-8 w-8 text-primary flex-shrink-0" aria-hidden="true" />
            <div className="flex-1">
              <p className="text-sm text-muted-foreground">Package Price</p>
              <div className="flex items-baseline gap-3 mt-1">
                <span className="text-2xl font-bold text-green-600">{formatCurrency(pkg.discountedPrice)}</span>
                {pkg.originalPrice > pkg.discountedPrice && (
                  <span className="text-lg text-muted-foreground line-through">
                    {formatCurrency(pkg.originalPrice)}
                  </span>
                )}
              </div>
            </div>
            <div className="text-right">
              <p className="text-sm text-muted-foreground">Duration</p>
              <p className="text-2xl font-bold">{pkg.totalDays}</p>
              <p className="text-sm text-muted-foreground">days</p>
            </div>
          </div>

          <Separator />

          {/* Location Details */}
          <div className="space-y-3">
            <div className="flex items-center gap-2">
              <MapPin className="h-5 w-5 text-primary" aria-hidden="true" />
              <h3 className="font-semibold text-lg">Location Details</h3>
            </div>
            <p className="text-muted-foreground leading-relaxed pl-7">{pkg.locationDetails}</p>
          </div>

          <Separator />

          {/* Itinerary Details */}
          <div className="space-y-3">
            <div className="flex items-center gap-2">
              <Calendar className="h-5 w-5 text-primary" aria-hidden="true" />
              <h3 className="font-semibold text-lg">Itinerary Details</h3>
            </div>
            <div className="text-muted-foreground leading-relaxed pl-7 whitespace-pre-line">{pkg.itineraryDetails}</div>
          </div>

          <Separator />

          {/* Other Details */}
          <div className="space-y-3">
            <div className="flex items-center gap-2">
              <Info className="h-5 w-5 text-primary" aria-hidden="true" />
              <h3 className="font-semibold text-lg">Inclusions & Exclusions</h3>
            </div>
            <div className="text-muted-foreground leading-relaxed pl-7 whitespace-pre-line">{pkg.otherDetails}</div>
          </div>

          <Separator />

          {/* Metadata */}
          <div className="grid grid-cols-2 gap-4 text-sm">
            <div>
              <p className="text-muted-foreground">Created</p>
              <p className="font-medium">{new Date(pkg.createdAt).toLocaleDateString("en-IN")}</p>
            </div>
            <div>
              <p className="text-muted-foreground">Last Updated</p>
              <p className="font-medium">{new Date(pkg.updatedAt).toLocaleDateString("en-IN")}</p>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  )
}
