"use client"

import { Card, CardContent } from "@/components/ui/card"
import { DollarSign, TrendingUp, Wallet, ArrowDownToLine } from "lucide-react"

interface EarningsSummaryCardsProps {
  totalEarning: number
  thisMonthEarning: number
  totalWalletBalance: number
  totalWithdrawn: number
  userType: "agent" | "hotel"
}

function formatCurrency(amount: number): string {
  if (amount >= 10000000) {
    return `₹${(amount / 10000000).toFixed(2)}Cr`
  }
  if (amount >= 100000) {
    return `₹${(amount / 100000).toFixed(2)}L`
  }
  return new Intl.NumberFormat("en-IN", {
    style: "currency",
    currency: "INR",
    maximumFractionDigits: 0,
  }).format(amount)
}

export function EarningsSummaryCards({
  totalEarning,
  thisMonthEarning,
  totalWalletBalance,
  totalWithdrawn,
  userType,
}: EarningsSummaryCardsProps) {
  const cards = [
    {
      title: "Total Earnings",
      value: formatCurrency(totalEarning),
      description: `All-time ${userType} earnings`,
      icon: DollarSign,
      color: "text-emerald-600",
      bgColor: "bg-emerald-50",
    },
    {
      title: "This Month",
      value: formatCurrency(thisMonthEarning),
      description: "Current month earnings",
      icon: TrendingUp,
      color: "text-blue-600",
      bgColor: "bg-blue-50",
    },
    {
      title: "Wallet Balance",
      value: formatCurrency(totalWalletBalance),
      description: "Available for withdrawal",
      icon: Wallet,
      color: "text-amber-600",
      bgColor: "bg-amber-50",
    },
    {
      title: "Total Withdrawn",
      value: formatCurrency(totalWithdrawn),
      description: "Successfully processed",
      icon: ArrowDownToLine,
      color: "text-purple-600",
      bgColor: "bg-purple-50",
    },
  ]

  return (
    <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
      {cards.map((card) => (
        <Card key={card.title}>
          <CardContent className="p-4">
            <div className="flex items-start justify-between">
              <div className="space-y-1">
                <p className="text-sm font-medium text-muted-foreground">{card.title}</p>
                <p className={`text-2xl font-bold ${card.color}`}>{card.value}</p>
                <p className="text-xs text-muted-foreground">{card.description}</p>
              </div>
              <div className={`rounded-lg p-2 ${card.bgColor}`}>
                <card.icon className={`h-5 w-5 ${card.color}`} aria-hidden="true" />
              </div>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  )
}
