"use client"

import { useState, useMemo, useCallback } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Plus } from "lucide-react"
import { CustomPackagesTable } from "./custom-packages-table"
import { DeletePackageDialog } from "./delete-package-dialog"
import { customPackageDataStore, type CustomPackage } from "@/lib/data/custom-packages"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"
import { useRouter } from "next/navigation"

export function CustomPackagesClient() {
  const toast = useToastEnhanced()
  const router = useRouter()
  const [currentTab, setCurrentTab] = useState("active")

  const [packages, setPackages] = useState<CustomPackage[]>(customPackageDataStore.getAllPackages())
  const [deletingPackage, setDeletingPackage] = useState<CustomPackage | null>(null)

  const activePackages = useMemo(() => packages.filter((p) => p.status === "active"), [packages])
  const inactivePackages = useMemo(() => packages.filter((p) => p.status === "inactive"), [packages])

  const handleTabChange = (value: string) => {
    setCurrentTab(value)
  }

  const handleToggleStatus = useCallback(
    (pkg: CustomPackage) => {
      const updated = customPackageDataStore.togglePackageStatus(pkg.id)
      if (updated) {
        setPackages(customPackageDataStore.getAllPackages())
        const action = updated.status === "active" ? "activated" : "deactivated"
        toast.success({
          title: `Package ${action.charAt(0).toUpperCase() + action.slice(1)}`,
          description: `${pkg.cityName} package has been ${action}.`,
        })
      }
    },
    [toast],
  )

  const handleDeletePackage = useCallback(() => {
    if (!deletingPackage) return

    const deleted = customPackageDataStore.deletePackage(deletingPackage.id)
    if (deleted) {
      setPackages(customPackageDataStore.getAllPackages())
      toast.success({
        title: "Package Deleted",
        description: `${deletingPackage.cityName} package has been deleted.`,
      })
    }
    setDeletingPackage(null)
  }, [deletingPackage, toast])

  const handleEdit = (pkg: CustomPackage) => {
    router.push(`/admin/custom-packages/edit/${pkg.id}`)
  }

  const getDisplayPackages = () => {
    switch (currentTab) {
      case "active":
        return activePackages
      case "inactive":
        return inactivePackages
      default:
        return packages
    }
  }

  return (
    <section aria-labelledby="custom-packages-heading" className="admin-section">
      <header className="admin-page-header flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 id="custom-packages-heading" className="admin-page-title">
            Custom Packages
          </h1>
          <p className="admin-page-description">Manage custom tour packages with multi-day itineraries.</p>
        </div>
        <Button onClick={() => router.push("/admin/custom-packages/add-new")} className="w-full sm:w-auto">
          <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
          Add New Package
        </Button>
      </header>

      <Tabs value={currentTab} onValueChange={handleTabChange} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activePackages.length})
          </TabsTrigger>
          <TabsTrigger
            value="inactive"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Inactive ({inactivePackages.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="active" className="mt-0">
          <CustomPackagesTable
            packages={getDisplayPackages()}
            onEdit={handleEdit}
            onDelete={setDeletingPackage}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="inactive" className="mt-0">
          <CustomPackagesTable
            packages={getDisplayPackages()}
            onEdit={handleEdit}
            onDelete={setDeletingPackage}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>
      </Tabs>

      <DeletePackageDialog
        package={deletingPackage}
        open={!!deletingPackage}
        onOpenChange={(open) => !open && setDeletingPackage(null)}
        onConfirm={handleDeletePackage}
      />
    </section>
  )
}
