"use client"

import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { Calendar, User, MapPin, Users, IndianRupee, Car, Clock, Tag } from "lucide-react"
import type { CustomPackageBooking } from "@/lib/data/custom-packages-bookings"
import { customPackageDataStore } from "@/lib/data/custom-packages"
import Image from "next/image"

interface ViewCustomBookingDetailsDialogProps {
  booking: CustomPackageBooking
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function ViewCustomBookingDetailsDialog({ booking, open, onOpenChange }: ViewCustomBookingDetailsDialogProps) {
  // Get full package details
  const packageDetails = customPackageDataStore.getPackageById(booking.packageId)

  // Format currency
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  // Format date
  const formatDate = (dateString: string): string => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      day: "2-digit",
      month: "long",
      year: "numeric",
    })
  }

  // Get status badge variant
  const getStatusVariant = (status: string) => {
    switch (status) {
      case "booked":
        return "default"
      case "ongoing":
        return "secondary"
      case "completed":
        return "outline"
      case "cancelled":
        return "destructive"
      default:
        return "default"
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="text-2xl">Custom Package Booking Details</DialogTitle>
          <DialogDescription>Complete information about this custom package booking</DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          {/* Status Badge */}
          <div className="flex items-center justify-between">
            <Badge variant={getStatusVariant(booking.status)} className="text-sm px-3 py-1 capitalize">
              {booking.status}
            </Badge>
            <span className="text-sm text-muted-foreground">Booking ID: {booking.id}</span>
          </div>

          <Separator />

          {/* Package Information */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <MapPin className="h-5 w-5 text-primary" />
              Package Information
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">City Name:</span>
                <span className="text-sm font-semibold">{booking.packageCityName}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Category:</span>
                <Badge variant="secondary" className="w-fit capitalize">
                  <Tag className="mr-1 h-3 w-3" />
                  {booking.packageCategory}
                </Badge>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Duration:</span>
                <span className="text-sm">
                  {booking.totalDays} Day{booking.totalDays > 1 ? "s" : ""}
                </span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Vehicle:</span>
                <span className="text-sm">{booking.selectedVehicle}</span>
              </div>
            </div>
          </div>

          {/* Customer Information */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <User className="h-5 w-5 text-primary" />
              Customer Information
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Customer Name:</span>
                <span className="text-sm font-semibold">{booking.customerName}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Customer ID:</span>
                <span className="text-sm">{booking.customerId}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Travelers Count:</span>
                <span className="text-sm flex items-center gap-1">
                  <Users className="h-4 w-4" />
                  {booking.travelersCount} person{booking.travelersCount > 1 ? "s" : ""}
                </span>
              </div>
            </div>
          </div>

          {/* Booking Timeline */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <Calendar className="h-5 w-5 text-primary" />
              Booking Timeline
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Booking Date:</span>
                <span className="text-sm">{formatDate(booking.bookingDate)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Travel Start:</span>
                <span className="text-sm font-semibold text-green-600">{formatDate(booking.startDate)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Travel End:</span>
                <span className="text-sm font-semibold text-red-600">{formatDate(booking.endDate)}</span>
              </div>
            </div>
          </div>

          {/* Payment Information */}
          <div>
            <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
              <IndianRupee className="h-5 w-5 text-primary" />
              Payment Information
            </h3>
            <div className="grid gap-3 bg-muted/30 p-4 rounded-lg">
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Total Amount:</span>
                <span className="text-sm font-bold text-lg">{formatCurrency(booking.bookingAmount)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Advance Paid:</span>
                <span className="text-sm text-green-600 font-semibold">{formatCurrency(booking.advanceAmount)}</span>
              </div>
              <div className="grid grid-cols-[140px_1fr] gap-2">
                <span className="text-sm font-medium text-muted-foreground">Pending Amount:</span>
                <span className="text-sm text-amber-600 font-semibold">{formatCurrency(booking.pendingAmount)}</span>
              </div>
            </div>
          </div>

          {/* Package Day Plans */}
          {packageDetails && packageDetails.dayPlans.length > 0 && (
            <div>
              <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
                <Clock className="h-5 w-5 text-primary" />
                Day-wise Itinerary
              </h3>
              <div className="space-y-4">
                {packageDetails.dayPlans.map((dayPlan) => (
                  <div key={dayPlan.dayNumber} className="bg-muted/30 p-4 rounded-lg space-y-3">
                    <h4 className="font-semibold text-base">Day {dayPlan.dayNumber}</h4>

                    {/* Locations */}
                    {dayPlan.locations.length > 0 && (
                      <div>
                        <p className="text-sm font-medium text-muted-foreground mb-2">Locations:</p>
                        <div className="grid gap-3">
                          {dayPlan.locations.map((location) => (
                            <div key={location.id} className="flex gap-3 p-3 bg-background rounded-lg border">
                              <Image
                                src={location.image || "/placeholder.svg"}
                                alt={location.name}
                                width={80}
                                height={60}
                                className="rounded object-cover"
                              />
                              <div className="flex-1">
                                <p className="text-sm font-medium">{location.name}</p>
                                <p className="text-xs text-muted-foreground mt-1">{location.itineraryDescription}</p>
                                {location.otherDetails && (
                                  <p className="text-xs text-muted-foreground mt-1">{location.otherDetails}</p>
                                )}
                              </div>
                            </div>
                          ))}
                        </div>
                      </div>
                    )}

                    {/* Vehicles */}
                    {dayPlan.vehicles.length > 0 && (
                      <div>
                        <p className="text-sm font-medium text-muted-foreground mb-2">Vehicle:</p>
                        <div className="flex items-center gap-3 p-3 bg-background rounded-lg border">
                          <Car className="h-4 w-4 text-muted-foreground" />
                          <div className="flex-1">
                            <p className="text-sm font-medium">{dayPlan.vehicles[0].vehicleName}</p>
                            <p className="text-xs text-muted-foreground">
                              {formatCurrency(dayPlan.vehicles[0].perDayPrice)}/day
                            </p>
                          </div>
                        </div>
                      </div>
                    )}

                    {/* Pricing */}
                    <div>
                      <p className="text-sm font-medium text-muted-foreground mb-2">Pricing:</p>
                      <div className="grid grid-cols-3 gap-2">
                        <div className="p-2 bg-background rounded border text-center">
                          <p className="text-xs text-muted-foreground">B2C Weekday</p>
                          <p className="text-sm font-semibold">{formatCurrency(dayPlan.pricing.b2cWeekdayPerHead)}</p>
                        </div>
                        <div className="p-2 bg-background rounded border text-center">
                          <p className="text-xs text-muted-foreground">B2C Weekend</p>
                          <p className="text-sm font-semibold">{formatCurrency(dayPlan.pricing.b2cWeekendPerHead)}</p>
                        </div>
                        <div className="p-2 bg-background rounded border text-center">
                          <p className="text-xs text-muted-foreground">B2B Agent</p>
                          <p className="text-sm font-semibold">{formatCurrency(dayPlan.pricing.b2bAgentPricing)}</p>
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  )
}
